<?php
/**
 * Authentication & Authorization System
 */

namespace Core;

use Core\Database;

class Auth {
    private $db;
    private $session;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->startSession();
    }
    
    private function startSession() {
        if (session_status() === PHP_SESSION_NONE) {
            $config = require __DIR__ . '/../config/app.php';
            session_name($config['session']['name']);
            session_set_cookie_params([
                'lifetime' => $config['session']['lifetime'],
                'path' => '/',
                'secure' => $config['session']['secure'],
                'httponly' => $config['session']['httponly']
            ]);
            session_start();
            
            // Regenerate session ID periodically
            if (!isset($_SESSION['created'])) {
                $_SESSION['created'] = time();
            } elseif (time() - $_SESSION['created'] > 1800) {
                session_regenerate_id(true);
                $_SESSION['created'] = time();
            }
        }
        $this->session = &$_SESSION;
    }
    
    public function login($username, $password) {
        $config = require __DIR__ . '/../config/app.php';
        
        // Check if user is locked
        $user = $this->db->fetch(
            "SELECT * FROM users WHERE username = ? OR email = ?",
            [$username, $username]
        );
        
        if (!$user) {
            return ['success' => false, 'message' => 'Invalid credentials'];
        }
        
        // Check lockout
        if ($user['locked_until'] && strtotime($user['locked_until']) > time()) {
            $remaining = ceil((strtotime($user['locked_until']) - time()) / 60);
            return ['success' => false, 'message' => "Account locked. Try again in {$remaining} minutes"];
        }
        
        // Verify password
        if (!password_verify($password, $user['password'])) {
            $attempts = $user['login_attempts'] + 1;
            
            if ($attempts >= $config['security']['login_attempts']) {
                $lockedUntil = date('Y-m-d H:i:s', time() + $config['security']['lockout_duration']);
                $this->db->query(
                    "UPDATE users SET login_attempts = ?, locked_until = ? WHERE id = ?",
                    [$attempts, $lockedUntil, $user['id']]
                );
                return ['success' => false, 'message' => 'Too many failed attempts. Account locked'];
            }
            
            $this->db->query(
                "UPDATE users SET login_attempts = ? WHERE id = ?",
                [$attempts, $user['id']]
            );
            
            return ['success' => false, 'message' => 'Invalid credentials'];
        }
        
        // Check if user is active
        if (!$user['is_active']) {
            return ['success' => false, 'message' => 'Account is disabled'];
        }
        
        // Successful login
        $this->db->query(
            "UPDATE users SET login_attempts = 0, locked_until = NULL, last_login = NOW() WHERE id = ?",
            [$user['id']]
        );
        
        // Load user with role and permissions
        $user['role'] = $this->db->fetch("SELECT * FROM roles WHERE id = ?", [$user['role_id']]);
        $user['permissions'] = json_decode($user['role']['permissions'] ?? '[]', true);
        
        // Set session
        $this->session['user_id'] = $user['id'];
        $this->session['user'] = $user;
        
        return ['success' => true, 'user' => $user];
    }
    
    public function logout() {
        session_unset();
        session_destroy();
        session_start();
    }
    
    public function check() {
        return isset($this->session['user_id']);
    }
    
    public function user() {
        if (!$this->check()) {
            return null;
        }
        
        if (!isset($this->session['user'])) {
            $userId = $this->session['user_id'];
            $user = $this->db->fetch(
                "SELECT u.*, r.name as role_name, r.permissions 
                 FROM users u 
                 LEFT JOIN roles r ON u.role_id = r.id 
                 WHERE u.id = ?",
                [$userId]
            );
            
            if ($user) {
                $user['permissions'] = json_decode($user['permissions'] ?? '[]', true);
                $this->session['user'] = $user;
            }
        }
        
        return $this->session['user'] ?? null;
    }
    
    public function hasPermission($permission) {
        $user = $this->user();
        if (!$user) return false;
        
        $permissions = $user['permissions'] ?? [];
        
        // Super admin has all permissions
        if (in_array('*', $permissions)) {
            return true;
        }
        
        // Check exact permission
        if (in_array($permission, $permissions)) {
            return true;
        }
        
        // Check wildcard permissions (e.g., "qr.*" matches "qr.create")
        foreach ($permissions as $perm) {
            if (strpos($perm, '*') !== false) {
                $pattern = str_replace('*', '.*', preg_quote($perm, '/'));
                if (preg_match("/^{$pattern}$/", $permission)) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    public function canAccessBranch($branchId) {
        $user = $this->user();
        if (!$user) return false;
        
        // Super admin can access all branches
        if ($this->hasPermission('*')) {
            return true;
        }
        
        // User can access their own branch
        return $user['branch_id'] == $branchId;
    }
    
    public function requireAuth() {
        if (!$this->check()) {
            Core\Helpers::redirect('login.php');
        }
    }
    
    public function requirePermission($permission) {
        $this->requireAuth();
        if (!$this->hasPermission($permission)) {
            http_response_code(403);
            die('Access Denied');
        }
    }
}
