<?php
/**
 * Dynamic QR Code Manager
 */

namespace Core;

use Core\Database;
use Core\Security;
// QR Code library will be installed via Composer
// For now, using simple placeholder - install endroid/qr-code package

class QRCode {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Generate unique short code
     */
    private function generateShortCode($length = 8) {
        $config = require __DIR__ . '/../config/app.php';
        $length = $config['qr']['short_url_length'] ?? $length;
        
        do {
            $code = strtoupper(substr(bin2hex(random_bytes($length / 2)), 0, $length));
            $exists = $this->db->fetch("SELECT id FROM qr_codes WHERE short_code = ?", [$code]);
        } while ($exists);
        
        return $code;
    }
    
    /**
     * Create new QR code
     */
    public function create($data) {
        $shortCode = $this->generateShortCode();
        
        $sql = "INSERT INTO qr_codes (
            short_code, branch_id, department_id, project_id, product_id, warehouse_id,
            title, description, target_type, target_url, redirect_rules,
            is_active, password_protected, password_hash, expires_at, scan_limit, created_by
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        
        $passwordHash = null;
        if (!empty($data['password'])) {
            $passwordHash = Security::hashPassword($data['password']);
        }
        
        $redirectRules = !empty($data['redirect_rules']) ? json_encode($data['redirect_rules']) : null;
        
        $this->db->query($sql, [
            $shortCode,
            $data['branch_id'],
            $data['department_id'] ?? null,
            $data['project_id'] ?? null,
            $data['product_id'] ?? null,
            $data['warehouse_id'] ?? null,
            $data['title'],
            $data['description'] ?? null,
            $data['target_type'],
            $data['target_url'],
            $redirectRules,
            $data['is_active'] ?? 1,
            !empty($data['password']) ? 1 : 0,
            $passwordHash,
            $data['expires_at'] ?? null,
            $data['scan_limit'] ?? null,
            $data['created_by']
        ]);
        
        $qrId = $this->db->lastInsertId();
        
        // Generate QR code image
        $this->generateQRImage($shortCode);
        
        // Log action
        $this->logAction('create', 'qr_code', $qrId, null, $data, $data['created_by']);
        
        return $this->getById($qrId);
    }
    
    /**
     * Update QR code
     */
    public function update($id, $data) {
        $oldData = $this->getById($id);
        
        $sql = "UPDATE qr_codes SET 
            title = ?, description = ?, target_url = ?, redirect_rules = ?,
            is_active = ?, password_protected = ?, password_hash = ?,
            expires_at = ?, scan_limit = ?, updated_at = NOW()
            WHERE id = ?";
        
        $passwordHash = $oldData['password_hash'];
        if (isset($data['password']) && !empty($data['password'])) {
            $passwordHash = Security::hashPassword($data['password']);
        } elseif (isset($data['password']) && empty($data['password'])) {
            $passwordHash = null;
        }
        
        $redirectRules = !empty($data['redirect_rules']) ? json_encode($data['redirect_rules']) : $oldData['redirect_rules'];
        
        $this->db->query($sql, [
            $data['title'] ?? $oldData['title'],
            $data['description'] ?? $oldData['description'],
            $data['target_url'] ?? $oldData['target_url'],
            $redirectRules,
            $data['is_active'] ?? $oldData['is_active'],
            !empty($passwordHash) ? 1 : 0,
            $passwordHash,
            $data['expires_at'] ?? $oldData['expires_at'],
            $data['scan_limit'] ?? $oldData['scan_limit'],
            $id
        ]);
        
        // Log action
        $userId = $_SESSION['user_id'] ?? null;
        $this->logAction('update', 'qr_code', $id, $oldData, $data, $userId);
        
        return $this->getById($id);
    }
    
    /**
     * Get QR code by ID
     */
    public function getById($id) {
        $sql = "SELECT q.*, b.name as branch_name, d.name as department_name,
                p.name as project_name, pr.name as product_name, w.name as warehouse_name
                FROM qr_codes q
                LEFT JOIN branches b ON q.branch_id = b.id
                LEFT JOIN departments d ON q.department_id = d.id
                LEFT JOIN projects p ON q.project_id = p.id
                LEFT JOIN products pr ON q.product_id = pr.id
                LEFT JOIN warehouses w ON q.warehouse_id = w.id
                WHERE q.id = ?";
        
        return $this->db->fetch($sql, [$id]);
    }
    
    /**
     * Get QR code by short code
     */
    public function getByShortCode($shortCode) {
        $sql = "SELECT q.*, b.name as branch_name, d.name as department_name,
                p.name as project_name, pr.name as product_name, w.name as warehouse_name
                FROM qr_codes q
                LEFT JOIN branches b ON q.branch_id = b.id
                LEFT JOIN departments d ON q.department_id = d.id
                LEFT JOIN projects p ON q.project_id = p.id
                LEFT JOIN products pr ON q.product_id = pr.id
                LEFT JOIN warehouses w ON q.warehouse_id = w.id
                WHERE q.short_code = ?";
        
        return $this->db->fetch($sql, [$shortCode]);
    }
    
    /**
     * Generate QR code image
     */
    public function generateQRImage($shortCode) {
        $config = require __DIR__ . '/../config/app.php';
        $baseUrl = Core\Helpers::baseUrl();
        $url = "{$baseUrl}/r/{$shortCode}";
        
        // Use Google Charts API as fallback, or install endroid/qr-code
        // For production, install: composer require endroid/qr-code
        $qrUrl = "https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=" . urlencode($url);
        
        $path = $config['paths']['qr_codes'] . $shortCode . '.png';
        if (!is_dir(dirname($path))) {
            mkdir(dirname($path), 0755, true);
        }
        
        // Download QR code from API
        $qrImage = @file_get_contents($qrUrl);
        if ($qrImage) {
            file_put_contents($path, $qrImage);
        }
        
        return $path;
    }
    
    /**
     * Handle QR code redirect with rules
     */
    public function handleRedirect($shortCode, $requestData = []) {
        $qr = $this->getByShortCode($shortCode);
        
        if (!$qr) {
            return ['redirect' => '/404', 'error' => 'QR code not found'];
        }
        
        // Check if active
        if (!$qr['is_active']) {
            return ['redirect' => '/disabled', 'error' => 'QR code is disabled'];
        }
        
        // Check expiration
        if ($qr['expires_at'] && strtotime($qr['expires_at']) < time()) {
            return ['redirect' => '/expired', 'error' => 'QR code has expired'];
        }
        
        // Check scan limit
        if ($qr['scan_limit'] && $qr['scan_count'] >= $qr['scan_limit']) {
            return ['redirect' => '/limit_reached', 'error' => 'Scan limit reached'];
        }
        
        // Record scan
        $this->recordScan($qr['id'], $requestData);
        
        // Apply redirect rules
        $redirectUrl = $this->applyRedirectRules($qr, $requestData);
        
        return ['redirect' => $redirectUrl];
    }
    
    /**
     * Apply redirect rules
     */
    private function applyRedirectRules($qr, $requestData) {
        $redirectRules = json_decode($qr['redirect_rules'] ?? '[]', true);
        
        if (empty($redirectRules)) {
            return $qr['target_url'];
        }
        
        // Sort rules by priority
        usort($redirectRules, function($a, $b) {
            return ($a['priority'] ?? 999) <=> ($b['priority'] ?? 999);
        });
        
        $country = $requestData['country'] ?? null;
        $deviceType = $requestData['device_type'] ?? null;
        
        foreach ($redirectRules as $rule) {
            $matches = true;
            
            if (isset($rule['country']) && $rule['country'] !== $country) {
                $matches = false;
            }
            
            if (isset($rule['device_type']) && $rule['device_type'] !== $deviceType) {
                $matches = false;
            }
            
            if ($matches && !empty($rule['url'])) {
                return $rule['url'];
            }
        }
        
        return $qr['target_url'];
    }
    
    /**
     * Record scan
     */
    public function recordScan($qrId, $requestData) {
        $config = require __DIR__ . '/../config/app.php';
        
        if (!$config['qr']['scan_tracking_enabled']) {
            return;
        }
        
        $this->db->query(
            "INSERT INTO qr_scans (qr_code_id, ip_address, user_agent, device_type, country, city, latitude, longitude)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
            [
                $qrId,
                $requestData['ip'] ?? Security::getClientIP(),
                $requestData['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? null,
                $requestData['device_type'] ?? Security::getDeviceType($_SERVER['HTTP_USER_AGENT'] ?? ''),
                $requestData['country'] ?? null,
                $requestData['city'] ?? null,
                $requestData['latitude'] ?? null,
                $requestData['longitude'] ?? null
            ]
        );
        
        // Update scan count
        $this->db->query(
            "UPDATE qr_codes SET scan_count = scan_count + 1 WHERE id = ?",
            [$qrId]
        );
    }
    
    /**
     * Bulk create QR codes
     */
    public function bulkCreate($template, $count) {
        $created = [];
        $this->db->beginTransaction();
        
        try {
            for ($i = 0; $i < $count; $i++) {
                $data = $template;
                $data['title'] = $template['title'] . ' #' . ($i + 1);
                $created[] = $this->create($data);
            }
            
            $this->db->commit();
            return $created;
        } catch (\Exception $e) {
            $this->db->rollback();
            throw $e;
        }
    }
    
    /**
     * Log action
     */
    private function logAction($action, $entityType, $entityId, $oldValues, $newValues, $userId) {
        $this->db->query(
            "INSERT INTO audit_logs (user_id, action, entity_type, entity_id, old_values, new_values, ip_address, user_agent)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
            [
                $userId,
                $action,
                $entityType,
                $entityId,
                $oldValues ? json_encode($oldValues) : null,
                $newValues ? json_encode($newValues) : null,
                Security::getClientIP(),
                $_SERVER['HTTP_USER_AGENT'] ?? null
            ]
        );
    }
    
    /**
     * List QR codes with filters
     */
    public function list($filters = [], $page = 1, $perPage = 20) {
        $offset = ($page - 1) * $perPage;
        $where = ['1=1'];
        $params = [];
        
        if (!empty($filters['branch_id'])) {
            $where[] = "q.branch_id = ?";
            $params[] = $filters['branch_id'];
        }
        
        if (!empty($filters['product_id'])) {
            $where[] = "q.product_id = ?";
            $params[] = $filters['product_id'];
        }
        
        if (!empty($filters['target_type'])) {
            $where[] = "q.target_type = ?";
            $params[] = $filters['target_type'];
        }
        
        if (isset($filters['is_active'])) {
            $where[] = "q.is_active = ?";
            $params[] = $filters['is_active'];
        }
        
        if (!empty($filters['search'])) {
            $where[] = "(q.title LIKE ? OR q.short_code LIKE ?)";
            $searchTerm = "%{$filters['search']}%";
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }
        
        $whereClause = implode(' AND ', $where);
        
        $sql = "SELECT q.*, b.name as branch_name, pr.name as product_name
                FROM qr_codes q
                LEFT JOIN branches b ON q.branch_id = b.id
                LEFT JOIN products pr ON q.product_id = pr.id
                WHERE {$whereClause}
                ORDER BY q.created_at DESC
                LIMIT ? OFFSET ?";
        
        $params[] = $perPage;
        $params[] = $offset;
        
        return $this->db->fetchAll($sql, $params);
    }
    
    /**
     * Get statistics
     */
    public function getStatistics($filters = []) {
        $where = ['1=1'];
        $params = [];
        
        if (!empty($filters['branch_id'])) {
            $where[] = "q.branch_id = ?";
            $params[] = $filters['branch_id'];
        }
        
        $whereClause = implode(' AND ', $where);
        
        $stats = $this->db->fetch(
            "SELECT 
                COUNT(*) as total_qr_codes,
                SUM(CASE WHEN q.is_active = 1 THEN 1 ELSE 0 END) as active_qr_codes,
                SUM(q.scan_count) as total_scans
             FROM qr_codes q
             WHERE {$whereClause}",
            $params
        );
        
        return $stats;
    }
}
