<?php
/**
 * Security Utilities
 * CSRF, XSS, SQL Injection Protection
 */

namespace Core;

class Security {
    private static $csrfToken = null;
    
    public static function generateCSRFToken() {
        if (self::$csrfToken === null) {
            if (!isset($_SESSION['csrf_token'])) {
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
            }
            self::$csrfToken = $_SESSION['csrf_token'];
        }
        return self::$csrfToken;
    }
    
    public static function validateCSRFToken($token) {
        if (!isset($_SESSION['csrf_token'])) {
            return false;
        }
        return hash_equals($_SESSION['csrf_token'], $token);
    }
    
    public static function getCSRFTokenField() {
        $token = self::generateCSRFToken();
        return '<input type="hidden" name="_token" value="' . htmlspecialchars($token, ENT_QUOTES, 'UTF-8') . '">';
    }
    
    public static function sanitizeInput($input) {
        if (is_array($input)) {
            return array_map([self::class, 'sanitizeInput'], $input);
        }
        return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
    }
    
    public static function sanitizeOutput($output) {
        return htmlspecialchars($output, ENT_QUOTES, 'UTF-8');
    }
    
    public static function validateInput($input, $rules) {
        $errors = [];
        
        foreach ($rules as $field => $ruleSet) {
            $value = $input[$field] ?? null;
            $rulesArray = explode('|', $ruleSet);
            
            foreach ($rulesArray as $rule) {
                $ruleParts = explode(':', $rule);
                $ruleName = $ruleParts[0];
                $ruleValue = $ruleParts[1] ?? null;
                
                switch ($ruleName) {
                    case 'required':
                        if (empty($value)) {
                            $errors[$field][] = "Field {$field} is required";
                        }
                        break;
                    
                    case 'email':
                        if ($value && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                            $errors[$field][] = "Field {$field} must be a valid email";
                        }
                        break;
                    
                    case 'min':
                        if ($value && strlen($value) < $ruleValue) {
                            $errors[$field][] = "Field {$field} must be at least {$ruleValue} characters";
                        }
                        break;
                    
                    case 'max':
                        if ($value && strlen($value) > $ruleValue) {
                            $errors[$field][] = "Field {$field} must not exceed {$ruleValue} characters";
                        }
                        break;
                    
                    case 'numeric':
                        if ($value && !is_numeric($value)) {
                            $errors[$field][] = "Field {$field} must be numeric";
                        }
                        break;
                }
            }
        }
        
        return empty($errors) ? true : $errors;
    }
    
    public static function hashPassword($password) {
        return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
    }
    
    public static function verifyPassword($password, $hash) {
        return password_verify($password, $hash);
    }
    
    public static function generateRandomString($length = 16) {
        return bin2hex(random_bytes($length / 2));
    }
    
    public static function getClientIP() {
        $ipKeys = ['HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR'];
        foreach ($ipKeys as $key) {
            if (array_key_exists($key, $_SERVER) === true) {
                foreach (explode(',', $_SERVER[$key]) as $ip) {
                    $ip = trim($ip);
                    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== false) {
                        return $ip;
                    }
                }
            }
        }
        return $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
    
    public static function getDeviceType($userAgent) {
        if (preg_match('/mobile|android|iphone|ipad/i', $userAgent)) {
            return 'mobile';
        } elseif (preg_match('/tablet|ipad/i', $userAgent)) {
            return 'tablet';
        }
        return 'desktop';
    }
}
