-- ============================================
-- Dynamic QR Code System - Database Schema
-- PHP 8.x + MySQL
-- ============================================

SET SQL_MODE = "NO_AUTO_VALUE_ON_ZERO";
SET time_zone = "+00:00";
SET NAMES utf8mb4;

-- ============================================
-- 1. Branches (الفروع)
-- ============================================
CREATE TABLE IF NOT EXISTS `branches` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `name` VARCHAR(100) NOT NULL,
  `code` VARCHAR(20) NOT NULL UNIQUE,
  `address` TEXT,
  `phone` VARCHAR(20),
  `email` VARCHAR(100),
  `is_active` TINYINT(1) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_code` (`code`),
  KEY `idx_active` (`is_active`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 2. Departments (الأقسام)
-- ============================================
CREATE TABLE IF NOT EXISTS `departments` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `branch_id` INT(11) UNSIGNED NOT NULL,
  `name` VARCHAR(100) NOT NULL,
  `code` VARCHAR(20) NOT NULL,
  `description` TEXT,
  `is_active` TINYINT(1) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`branch_id`) REFERENCES `branches`(`id`) ON DELETE RESTRICT,
  KEY `idx_branch` (`branch_id`),
  KEY `idx_code` (`code`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 3. Roles (الأدوار)
-- ============================================
CREATE TABLE IF NOT EXISTS `roles` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `name` VARCHAR(50) NOT NULL UNIQUE,
  `display_name` VARCHAR(100) NOT NULL,
  `description` TEXT,
  `permissions` JSON,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_name` (`name`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 4. Users (المستخدمون)
-- ============================================
CREATE TABLE IF NOT EXISTS `users` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `branch_id` INT(11) UNSIGNED,
  `department_id` INT(11) UNSIGNED,
  `role_id` INT(11) UNSIGNED NOT NULL,
  `username` VARCHAR(50) NOT NULL UNIQUE,
  `email` VARCHAR(100) NOT NULL UNIQUE,
  `password` VARCHAR(255) NOT NULL,
  `full_name` VARCHAR(100) NOT NULL,
  `phone` VARCHAR(20),
  `is_active` TINYINT(1) DEFAULT 1,
  `last_login` TIMESTAMP NULL,
  `login_attempts` INT(3) DEFAULT 0,
  `locked_until` TIMESTAMP NULL,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`branch_id`) REFERENCES `branches`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`department_id`) REFERENCES `departments`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`role_id`) REFERENCES `roles`(`id`) ON DELETE RESTRICT,
  KEY `idx_username` (`username`),
  KEY `idx_email` (`email`),
  KEY `idx_branch` (`branch_id`),
  KEY `idx_role` (`role_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 5. Projects / Campaigns (المشاريع)
-- ============================================
CREATE TABLE IF NOT EXISTS `projects` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `branch_id` INT(11) UNSIGNED NOT NULL,
  `name` VARCHAR(100) NOT NULL,
  `code` VARCHAR(50) NOT NULL,
  `type` ENUM('product', 'supplier', 'customer') DEFAULT 'product',
  `description` TEXT,
  `start_date` DATE,
  `end_date` DATE,
  `is_active` TINYINT(1) DEFAULT 1,
  `created_by` INT(11) UNSIGNED,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`branch_id`) REFERENCES `branches`(`id`) ON DELETE RESTRICT,
  FOREIGN KEY (`created_by`) REFERENCES `users`(`id`) ON DELETE SET NULL,
  KEY `idx_branch` (`branch_id`),
  KEY `idx_code` (`code`),
  KEY `idx_type` (`type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 6. Products (المنتجات)
-- ============================================
CREATE TABLE IF NOT EXISTS `products` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `branch_id` INT(11) UNSIGNED NOT NULL,
  `sku` VARCHAR(100) NOT NULL,
  `name` VARCHAR(200) NOT NULL,
  `model` VARCHAR(100),
  `type` ENUM('motor', 'gearbox', 'other') DEFAULT 'motor',
  `description` TEXT,
  `specifications` JSON,
  `datasheet_pdf` VARCHAR(255),
  `catalog_pdf` VARCHAR(255),
  `installation_guide` TEXT,
  `safety_warnings` TEXT,
  `video_url` VARCHAR(500),
  `is_active` TINYINT(1) DEFAULT 1,
  `created_by` INT(11) UNSIGNED,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`branch_id`) REFERENCES `branches`(`id`) ON DELETE RESTRICT,
  FOREIGN KEY (`created_by`) REFERENCES `users`(`id`) ON DELETE SET NULL,
  UNIQUE KEY `unique_sku_branch` (`sku`, `branch_id`),
  KEY `idx_sku` (`sku`),
  KEY `idx_branch` (`branch_id`),
  KEY `idx_type` (`type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 7. Warehouses / Storage Locations (المخازن)
-- ============================================
CREATE TABLE IF NOT EXISTS `warehouses` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `branch_id` INT(11) UNSIGNED NOT NULL,
  `code` VARCHAR(50) NOT NULL,
  `name` VARCHAR(100) NOT NULL,
  `location` VARCHAR(200),
  `area` VARCHAR(100),
  `rack` VARCHAR(50),
  `zone` VARCHAR(50),
  `description` TEXT,
  `is_active` TINYINT(1) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`branch_id`) REFERENCES `branches`(`id`) ON DELETE RESTRICT,
  UNIQUE KEY `unique_code_branch` (`code`, `branch_id`),
  KEY `idx_branch` (`branch_id`),
  KEY `idx_code` (`code`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 8. Inventory (المخزون)
-- ============================================
CREATE TABLE IF NOT EXISTS `inventory` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `warehouse_id` INT(11) UNSIGNED NOT NULL,
  `product_id` INT(11) UNSIGNED NOT NULL,
  `quantity` INT(11) DEFAULT 0,
  `min_threshold` INT(11) DEFAULT 0,
  `status` ENUM('in_stock', 'low_stock', 'out_of_stock') DEFAULT 'in_stock',
  `last_counted` TIMESTAMP NULL,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouses`(`id`) ON DELETE RESTRICT,
  FOREIGN KEY (`product_id`) REFERENCES `products`(`id`) ON DELETE RESTRICT,
  UNIQUE KEY `unique_warehouse_product` (`warehouse_id`, `product_id`),
  KEY `idx_warehouse` (`warehouse_id`),
  KEY `idx_product` (`product_id`),
  KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 9. QR Codes (رموز QR)
-- ============================================
CREATE TABLE IF NOT EXISTS `qr_codes` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `short_code` VARCHAR(20) NOT NULL UNIQUE,
  `branch_id` INT(11) UNSIGNED NOT NULL,
  `department_id` INT(11) UNSIGNED,
  `project_id` INT(11) UNSIGNED,
  `product_id` INT(11) UNSIGNED,
  `warehouse_id` INT(11) UNSIGNED,
  `title` VARCHAR(200) NOT NULL,
  `description` TEXT,
  `target_type` ENUM('product', 'warehouse', 'invoice', 'warranty', 'custom') NOT NULL,
  `target_url` TEXT NOT NULL,
  `redirect_rules` JSON,
  `is_active` TINYINT(1) DEFAULT 1,
  `password_protected` TINYINT(1) DEFAULT 0,
  `password_hash` VARCHAR(255),
  `expires_at` TIMESTAMP NULL,
  `scan_limit` INT(11) NULL,
  `scan_count` INT(11) DEFAULT 0,
  `created_by` INT(11) UNSIGNED NOT NULL,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`branch_id`) REFERENCES `branches`(`id`) ON DELETE RESTRICT,
  FOREIGN KEY (`department_id`) REFERENCES `departments`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`project_id`) REFERENCES `projects`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`product_id`) REFERENCES `products`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouses`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`created_by`) REFERENCES `users`(`id`) ON DELETE RESTRICT,
  KEY `idx_short_code` (`short_code`),
  KEY `idx_branch` (`branch_id`),
  KEY `idx_product` (`product_id`),
  KEY `idx_warehouse` (`warehouse_id`),
  KEY `idx_active` (`is_active`),
  KEY `idx_expires` (`expires_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 10. QR Scans (سجلات المسح)
-- ============================================
CREATE TABLE IF NOT EXISTS `qr_scans` (
  `id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
  `qr_code_id` INT(11) UNSIGNED NOT NULL,
  `ip_address` VARCHAR(45),
  `user_agent` VARCHAR(500),
  `device_type` VARCHAR(50),
  `country` VARCHAR(2),
  `city` VARCHAR(100),
  `latitude` DECIMAL(10, 8),
  `longitude` DECIMAL(11, 8),
  `scanned_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`qr_code_id`) REFERENCES `qr_codes`(`id`) ON DELETE CASCADE,
  KEY `idx_qr_code` (`qr_code_id`),
  KEY `idx_scanned_at` (`scanned_at`),
  KEY `idx_country` (`country`),
  KEY `idx_device` (`device_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 11. Audit Logs (سجلات المراجعة)
-- ============================================
CREATE TABLE IF NOT EXISTS `audit_logs` (
  `id` BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
  `user_id` INT(11) UNSIGNED,
  `action` VARCHAR(100) NOT NULL,
  `entity_type` VARCHAR(50),
  `entity_id` INT(11) UNSIGNED,
  `old_values` JSON,
  `new_values` JSON,
  `ip_address` VARCHAR(45),
  `user_agent` VARCHAR(500),
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE SET NULL,
  KEY `idx_user` (`user_id`),
  KEY `idx_action` (`action`),
  KEY `idx_entity` (`entity_type`, `entity_id`),
  KEY `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 12. Inventory Alerts (تنبيهات المخزون)
-- ============================================
CREATE TABLE IF NOT EXISTS `inventory_alerts` (
  `id` INT(11) UNSIGNED NOT NULL AUTO_INCREMENT,
  `inventory_id` INT(11) UNSIGNED NOT NULL,
  `warehouse_id` INT(11) UNSIGNED NOT NULL,
  `product_id` INT(11) UNSIGNED NOT NULL,
  `alert_type` ENUM('low_stock', 'out_of_stock', 'scanned_report') NOT NULL,
  `message` TEXT,
  `is_resolved` TINYINT(1) DEFAULT 0,
  `resolved_by` INT(11) UNSIGNED,
  `resolved_at` TIMESTAMP NULL,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  FOREIGN KEY (`inventory_id`) REFERENCES `inventory`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`warehouse_id`) REFERENCES `warehouses`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`product_id`) REFERENCES `products`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`resolved_by`) REFERENCES `users`(`id`) ON DELETE SET NULL,
  KEY `idx_inventory` (`inventory_id`),
  KEY `idx_resolved` (`is_resolved`),
  KEY `idx_alert_type` (`alert_type`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Insert Default Data
-- ============================================

-- Default Roles
INSERT INTO `roles` (`name`, `display_name`, `description`, `permissions`) VALUES
('super_admin', 'Super Admin', 'Full system access', '["*"]'),
('admin', 'Admin', 'Branch admin access', '["qr.create", "qr.edit", "qr.delete", "products.*", "warehouses.*", "reports.view"]'),
('manager', 'Manager', 'Department manager', '["qr.create", "qr.edit", "products.view", "warehouses.view", "reports.view"]'),
('staff', 'Staff', 'Regular staff member', '["qr.view", "products.view"]'),
('viewer', 'Viewer', 'Read-only access', '["qr.view", "products.view", "reports.view"]');

-- Default Branch
INSERT INTO `branches` (`name`, `code`, `address`, `is_active`) VALUES
('الفرع الرئيسي', 'MAIN', 'العنوان الرئيسي', 1);

-- Default Departments
INSERT INTO `departments` (`branch_id`, `name`, `code`, `description`) VALUES
(1, 'المبيعات', 'SALES', 'قسم المبيعات'),
(1, 'المخازن', 'WAREHOUSE', 'قسم المخازن'),
(1, 'الصيانة', 'MAINTENANCE', 'قسم الصيانة'),
(1, 'الإدارة', 'ADMIN', 'الإدارة العليا');

-- Default Super Admin User (password: admin123 - change after installation)
-- Note: This password hash is for 'admin123'. If it doesn't work, run reset_password.php
INSERT INTO `users` (`branch_id`, `department_id`, `role_id`, `username`, `email`, `password`, `full_name`, `is_active`) VALUES
(1, 4, 1, 'admin', 'admin@company.com', '$2y$12$LQv3c1yqBWVHxkd0LHAkCOYz6TtxMQJqhN8/LewY5rGqgOdZPrGhG', 'Super Admin', 1);
