<?php
/**
 * View QR Code Details
 */

require_once __DIR__ . '/../core/Autoloader.php';

use Core\Auth;
use Core\QRCode;
use Core\Security;
use Core\Database;

$auth = new Auth();
$auth->requireAuth();

$qrHandler = new QRCode();
$db = Database::getInstance();

$id = $_GET['id'] ?? 0;
$qr = $qrHandler->getById($id);

if (!$qr) {
    Core\Helpers::redirect('404.php');
}

// Check access
if (!$auth->hasPermission('*') && $qr['branch_id'] != $auth->user()['branch_id']) {
    Core\Helpers::redirect('403.php');
}

// Get scan statistics
$scanStats = $db->fetch(
    "SELECT 
        COUNT(*) as total_scans,
        COUNT(DISTINCT DATE(scanned_at)) as unique_days,
        COUNT(DISTINCT ip_address) as unique_ips
     FROM qr_scans 
     WHERE qr_code_id = ?",
    [$id]
);

// Get recent scans
$recentScans = $db->fetchAll(
    "SELECT * FROM qr_scans 
     WHERE qr_code_id = ? 
     ORDER BY scanned_at DESC 
     LIMIT 20",
    [$id]
);

$config = require __DIR__ . '/../config/app.php';
$qrUrl = $config['base_url'] . '/r/' . $qr['short_code'];
$qrImagePath = $config['paths']['qr_codes'] . $qr['short_code'] . '.png';

$pageTitle = 'تفاصيل رمز QR';
include '../includes/header.php';
?>

<div class="page-header">
    <h1><?= Security::sanitizeOutput($qr['title']) ?></h1>
    <div>
        <?php if ($auth->hasPermission('qr.edit')): ?>
            <a href="<?= Core\Helpers::url('qr/edit.php?id=' . $qr['id']) ?>" class="btn btn-secondary">تعديل</a>
        <?php endif; ?>
        <a href="<?= Core\Helpers::url('qr/list.php') ?>" class="btn btn-secondary">رجوع</a>
    </div>
</div>

<div class="grid-2">
    <div class="card">
        <h2>معلومات رمز QR</h2>
        <table class="info-table">
            <tr>
                <th>الكود القصير:</th>
                <td><code><?= Security::sanitizeOutput($qr['short_code']) ?></code></td>
            </tr>
            <tr>
                <th>الرابط:</th>
                <td><a href="<?= $qrUrl ?>" target="_blank"><?= $qrUrl ?></a></td>
            </tr>
            <tr>
                <th>العنوان:</th>
                <td><?= Security::sanitizeOutput($qr['title']) ?></td>
            </tr>
            <tr>
                <th>النوع:</th>
                <td><?= Security::sanitizeOutput($qr['target_type']) ?></td>
            </tr>
            <tr>
                <th>الرابط الهدف:</th>
                <td><a href="<?= $qr['target_url'] ?>" target="_blank"><?= Security::sanitizeOutput($qr['target_url']) ?></a></td>
            </tr>
            <tr>
                <th>الفرع:</th>
                <td><?= Security::sanitizeOutput($qr['branch_name'] ?? '-') ?></td>
            </tr>
            <tr>
                <th>القسم:</th>
                <td><?= Security::sanitizeOutput($qr['department_name'] ?? '-') ?></td>
            </tr>
            <tr>
                <th>المنتج:</th>
                <td><?= Security::sanitizeOutput($qr['product_name'] ?? '-') ?></td>
            </tr>
            <tr>
                <th>الحالة:</th>
                <td>
                    <span class="badge <?= $qr['is_active'] ? 'badge-success' : 'badge-danger' ?>">
                        <?= $qr['is_active'] ? 'نشط' : 'غير نشط' ?>
                    </span>
                </td>
            </tr>
            <tr>
                <th>عدد المسحات:</th>
                <td><?= number_format($qr['scan_count']) ?></td>
            </tr>
            <?php if ($qr['expires_at']): ?>
            <tr>
                <th>تاريخ الانتهاء:</th>
                <td><?= date('Y-m-d H:i', strtotime($qr['expires_at'])) ?></td>
            </tr>
            <?php endif; ?>
        </table>
    </div>
    
    <div class="card">
        <h2>صورة رمز QR</h2>
        <?php if (file_exists($qrImagePath)): ?>
            <div class="qr-image-container">
                <img src="<?= Core\Helpers::url('uploads/qr_codes/' . $qr['short_code'] . '.png') ?>" alt="QR Code" class="qr-image">
                <div class="qr-actions">
                    <a href="<?= Core\Helpers::url('uploads/qr_codes/' . $qr['short_code'] . '.png') ?>" download class="btn btn-primary">تحميل</a>
                    <button onclick="window.print()" class="btn btn-secondary">طباعة</button>
                </div>
            </div>
        <?php else: ?>
            <p>جارٍ إنشاء صورة رمز QR...</p>
        <?php endif; ?>
    </div>
</div>

<div class="card">
    <h2>إحصائيات المسح</h2>
    <div class="stats-grid">
        <div class="stat-item">
            <strong><?= number_format($scanStats['total_scans']) ?></strong>
            <span>إجمالي المسحات</span>
        </div>
        <div class="stat-item">
            <strong><?= number_format($scanStats['unique_ips']) ?></strong>
            <span>عنوان IP مختلف</span>
        </div>
        <div class="stat-item">
            <strong><?= number_format($scanStats['unique_days']) ?></strong>
            <span>أيام مختلفة</span>
        </div>
    </div>
</div>

<div class="card">
    <h2>آخر المسحات</h2>
    <div class="table-container">
        <table class="data-table">
            <thead>
                <tr>
                    <th>التاريخ</th>
                    <th>عنوان IP</th>
                    <th>نوع الجهاز</th>
                    <th>الدولة</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($recentScans)): ?>
                    <tr>
                        <td colspan="4" class="text-center">لا توجد مسحات بعد</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($recentScans as $scan): ?>
                        <tr>
                            <td><?= date('Y-m-d H:i:s', strtotime($scan['scanned_at'])) ?></td>
                            <td><?= Security::sanitizeOutput($scan['ip_address']) ?></td>
                            <td><?= Security::sanitizeOutput($scan['device_type'] ?? '-') ?></td>
                            <td><?= Security::sanitizeOutput($scan['country'] ?? '-') ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
