<?php
/**
 * Reports Dashboard
 */

require_once __DIR__ . '/../core/Autoloader.php';

use Core\Auth;
use Core\Security;
use Core\Database;

$auth = new Auth();
$auth->requirePermission('reports.view');

$db = Database::getInstance();
$user = $auth->user();

// Get date range
$startDate = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$endDate = $_GET['end_date'] ?? date('Y-m-d');

// Build where clause
$where = [];
$params = [];

if (!$auth->hasPermission('*')) {
    $where[] = "q.branch_id = ?";
    $params[] = $user['branch_id'];
}

$where[] = "DATE(s.scanned_at) BETWEEN ? AND ?";
$params[] = $startDate;
$params[] = $endDate;

$whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

// Overall statistics
$overallStats = $db->fetch(
    "SELECT 
        COUNT(DISTINCT s.qr_code_id) as active_qr_codes,
        COUNT(*) as total_scans,
        COUNT(DISTINCT s.ip_address) as unique_visitors,
        COUNT(DISTINCT DATE(s.scanned_at)) as active_days
     FROM qr_scans s
     LEFT JOIN qr_codes q ON s.qr_code_id = q.id
     {$whereClause}",
    $params
);

// Top QR codes
$topQRCodes = $db->fetchAll(
    "SELECT q.id, q.short_code, q.title, COUNT(*) as scan_count
     FROM qr_scans s
     LEFT JOIN qr_codes q ON s.qr_code_id = q.id
     {$whereClause}
     GROUP BY q.id, q.short_code, q.title
     ORDER BY scan_count DESC
     LIMIT 10",
    $params
);

// Scans by device type
$scansByDevice = $db->fetchAll(
    "SELECT device_type, COUNT(*) as count
     FROM qr_scans s
     LEFT JOIN qr_codes q ON s.qr_code_id = q.id
     {$whereClause}
     GROUP BY device_type
     ORDER BY count DESC",
    $params
);

// Scans by day
$scansByDay = $db->fetchAll(
    "SELECT DATE(scanned_at) as date, COUNT(*) as count
     FROM qr_scans s
     LEFT JOIN qr_codes q ON s.qr_code_id = q.id
     {$whereClause}
     GROUP BY DATE(scanned_at)
     ORDER BY date ASC",
    $params
);

$pageTitle = 'لوحة التقارير';
include '../includes/header.php';
?>

<div class="page-header">
    <h1>لوحة التقارير</h1>
</div>

<div class="filters-card">
    <form method="GET" class="filter-form">
        <div class="form-group">
            <label>من:</label>
            <input type="date" name="start_date" value="<?= $startDate ?>">
        </div>
        <div class="form-group">
            <label>إلى:</label>
            <input type="date" name="end_date" value="<?= $endDate ?>">
        </div>
        <button type="submit" class="btn btn-primary">تطبيق</button>
    </form>
</div>

<div class="stats-grid">
    <div class="stat-card">
        <div class="stat-icon">📊</div>
        <div class="stat-content">
            <h3><?= number_format($overallStats['total_scans']) ?></h3>
            <p>إجمالي المسحات</p>
        </div>
    </div>
    
    <div class="stat-card">
        <div class="stat-icon">📱</div>
        <div class="stat-content">
            <h3><?= number_format($overallStats['active_qr_codes']) ?></h3>
            <p>رموز QR نشطة</p>
        </div>
    </div>
    
    <div class="stat-card">
        <div class="stat-icon">👥</div>
        <div class="stat-content">
            <h3><?= number_format($overallStats['unique_visitors']) ?></h3>
            <p>زوار مختلفون</p>
        </div>
    </div>
    
    <div class="stat-card">
        <div class="stat-icon">📅</div>
        <div class="stat-content">
            <h3><?= number_format($overallStats['active_days']) ?></h3>
            <p>أيام نشطة</p>
        </div>
    </div>
</div>

<div class="grid-2">
    <div class="card">
        <h2>أكثر رموز QR استخداماً</h2>
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>الكود</th>
                        <th>العنوان</th>
                        <th>عدد المسحات</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($topQRCodes as $qr): ?>
                        <tr>
                            <td><code><?= Security::sanitizeOutput($qr['short_code']) ?></code></td>
                            <td><?= Security::sanitizeOutput($qr['title']) ?></td>
                            <td><?= number_format($qr['scan_count']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <div class="card">
        <h2>المسحات حسب نوع الجهاز</h2>
        <div class="table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>نوع الجهاز</th>
                        <th>عدد المسحات</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($scansByDevice as $device): ?>
                        <tr>
                            <td><?= Security::sanitizeOutput($device['device_type'] ?? 'غير محدد') ?></td>
                            <td><?= number_format($device['count']) ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<div class="card">
    <h2>المسحات حسب التاريخ</h2>
    <canvas id="scansChart" width="400" height="100"></canvas>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
const scansData = <?= json_encode($scansByDay) ?>;
const ctx = document.getElementById('scansChart').getContext('2d');
new Chart(ctx, {
    type: 'line',
    data: {
        labels: scansData.map(d => d.date),
        datasets: [{
            label: 'عدد المسحات',
            data: scansData.map(d => d.count),
            borderColor: 'rgb(75, 192, 192)',
            tension: 0.1
        }]
    },
    options: {
        responsive: true,
        scales: {
            y: {
                beginAtZero: true
            }
        }
    }
});
</script>

<?php include '../includes/footer.php'; ?>
